#ifndef _SSH_SSH_h
#define _SSH_SSH_h

#include <Core/Core.h>
#include <JobQueue/JobQueue.h>

#include "libssh2/libssh2.h"
#include "libssh2/libssh2_sftp.h"
#include "libssh2/libssh2_publickey.h"

namespace Upp {

INITIALIZE(SSH)

typedef LIBSSH2_SESSION         SshSession;
typedef libssh2_knownhost       SshHost;
typedef LIBSSH2_CHANNEL         SshChannel;
typedef LIBSSH2_SFTP            SFtpSession;
typedef LIBSSH2_SFTP_HANDLE     SFtpHandle;
typedef LIBSSH2_SFTP_ATTRIBUTES SFtpAttrs;

class Ssh : private NoCopy, public JobQueue {
    friend class Subsystem;
    friend class KnownHosts;
public:
    class Subsystem : NoCopy, public JobQueue {
        public:
            struct TypeError : public Exc {
                TypeError(const String& err) : Exc(err) {}
            };
            enum class Type     { UNDEFINED, CHANNEL, SFTP, SCP, EXEC, SHELL };
            void                Session(Ssh& session);
            void                ChunkSize(size_t sz)        { if(sz > 4096) chunk_size = sz; }
            Type                GetType() const             { return type; }
            template <class T>  T&   To()                   { auto* t = dynamic_cast<T*>(this); if(!t) throw TypeError("Invalid object conversion"); return *t; }
            template <class T>  bool Is() const             { return dynamic_cast<const T*>(this); }
            void                Clear();
            Subsystem();
            virtual ~Subsystem()                            { ssh = NULL; }

        protected:
            virtual void        StartInit() = 0;
            virtual void        StartStop() = 0;
            virtual void        CleanUp()   = 0;
            void                Error(int code = 0, const char* msg = NULL);
            bool                WouldBlock() const          { ASSERT(ssh); return ssh->WouldBlock(); }

            Ssh*                ssh;
            Type                type;
            size_t              chunk_size;
            String              packet;
            int64               packet_length;
    };

public:
    enum Methods {
        METHOD_EXCHANGE = 0,
        METHOD_HOSTKEY,
        METHOD_CENCRYPTION,
        METHOD_SENCRYPTION,
        METHOD_CMAC,
        METHOD_SMAC,
        METHOD_CCOMPRESSION,
        METHOD_SCOMPRESSION
    };

    Ssh&                      Auth(const String& user, const String& pass)                                      { username = user; password = pass; return *this; }
    Ssh&                      Keys(const String& prikey, const String& pubkey, const String& phrase = Null)     { private_key = prikey; public_key = pubkey; return *this; }
    Ssh&                      Method(int type, Value method)                                                    { ssh_methods(type) = method; return *this; }
    Ssh&                      Methods(ValueMap methods)                                                         { ssh_methods = clone(methods); return *this; }

    Ssh&                      Password()                                                                        { auth_method = PASSWORD;  return *this; }
    Ssh&                      PublicKey()                                                                       { auth_method = PUBLICKEY; return *this; }
    Ssh&                      Interactive()                                                                     { auth_method = KEYBOARD;  return *this; }

    TcpSocket&                GetSocket()                                                                       { return socket; }
    SshSession*               GetSession() const                                                                { return session; }
    String                    GetBanner() const                                                                 { return session ? String(libssh2_session_banner_get(session)) : Null; }
    const String&             GetFingerprint() const                                                            { return fingerprint; }
    String                    GetMethod(int type)                                                               { return session ? String(libssh2_session_methods(session, type)) : Null; }
    ValueMap                  GetMethods();
    Vector<String>            GetAuthMethods()                                                                  { return pick(Split(auth_methods, ',')); }

    Ssh&                      StartConnect(const String& host, int port);
    Ssh&                      StartConnect(const String& host, int port, const String& user, const String& pass){ return Auth(user, pass).StartConnect(host, port); }
    Ssh&                      StartDisconnect();
    bool                      Connect(const String& host, int port)                                             { ClearQueue(); StartConnect(host, port); return Execute(); }
    bool                      Connect(const String& host, int port, const String& user, const String& pass)     { ClearQueue(); Auth(user, pass).StartConnect(host, port); return Execute(); }
    bool                      Disconnect()                                                                      { ClearQueue(); StartDisconnect(); return Execute(); }

    Event<Ssh&>               WhenConfig;
    Event<Ssh&>               WhenAuth;
    Gate<Ssh&>                WhenVerify;
    Function<String(Ssh&, String, String, String)>  WhenKeyboard;

private:
    TcpSocket                 socket;
    IpAddrInfo                ip_addrinfo;
    SshSession*               session;
    String                    fingerprint;
    int                       key_type;
    int                       auth_method;
    String                    auth_methods;
    ValueMap                  ssh_methods;
    String                    username;
    String                    password;
    String                    passphrase;
    String                    private_key;
    String                    public_key;
    bool                      connected;

    void                      SetMethods(int method_type, const String& list);
    String                    GetMethodNames(int type);
    inline void               CheckAuthMethods();

    inline int                DoPwdAuth();
    inline int                DoKeyAuth();
    inline int                DoKbdAuth();

    void                      CleanUp();
    void                      Error(int code = 0, const char* msg = NULL);
    bool                      WouldBlock()                                      { return session && libssh2_session_last_errno(session) == LIBSSH2_ERROR_EAGAIN; }

    enum AuthMethod  { PASSWORD, PUBLICKEY, KEYBOARD };
    enum HostkeyType { RSAKEY, DSSKEY };

public:
    Ssh();
    Ssh(ValueMap methods) : Ssh() { Methods(methods); }
    virtual ~Ssh() { CleanUp(); }
};

class SFtp : public Ssh::Subsystem {
public:
    enum Flags {
        READ      = LIBSSH2_FXF_READ,
        WRITE     = LIBSSH2_FXF_WRITE,
        APPEND    = LIBSSH2_FXF_APPEND,
        CREATE    = LIBSSH2_FXF_CREAT,
        TRUNCATE  = LIBSSH2_FXF_TRUNC,
        EXCLUSIVE = LIBSSH2_FXF_EXCL
    };

    enum Permissions {
        IRUSR = LIBSSH2_SFTP_S_IRUSR,
        IWUSR = LIBSSH2_SFTP_S_IWUSR,
        IXUSR = LIBSSH2_SFTP_S_IXUSR,
        IRWXU = IRUSR | IWUSR | IXUSR,
        IRGRP = LIBSSH2_SFTP_S_IRGRP,
        IWGRP = LIBSSH2_SFTP_S_IWGRP,
        IXGRP = LIBSSH2_SFTP_S_IXGRP,
        IRWXG = IRGRP | IWGRP | IXGRP,
        IROTH = LIBSSH2_SFTP_S_IROTH,
        IWOTH = LIBSSH2_SFTP_S_IWOTH,
        IXOTH = LIBSSH2_SFTP_S_IXOTH,
        IRWXO = IROTH | IWOTH | IXOTH,
        RWXA  = IRWXU | IRWXG | IRWXO
    };

    class DirEntry : Moveable<DirEntry> {
        friend class SFtp;
        public:
            String GetName() const          { return file; }
            String GetEntry() const         { return entry; }
            int64  GetUid() const           { return a.flags & LIBSSH2_SFTP_ATTR_UIDGID ? a.uid : -1; }
            int64  GetGid() const           { return a.flags & LIBSSH2_SFTP_ATTR_UIDGID ? a.gid : -1; }
            int64  GetSize() const          { return a.flags & LIBSSH2_SFTP_ATTR_SIZE ? a.filesize : -1; }
            Time   GetLastModified() const  { return a.flags & LIBSSH2_SFTP_ATTR_ACMODTIME ? TimeFromUTC(a.mtime) : Null; }
            Time   GetLastAccessed() const  { return a.flags & LIBSSH2_SFTP_ATTR_ACMODTIME ? TimeFromUTC(a.atime) : Null; }
            SFtpAttrs& GetAttrs()           { return a; }

            bool IsFile() const             { return LIBSSH2_SFTP_S_ISREG(a.permissions); }
            bool IsDirectory() const        { return LIBSSH2_SFTP_S_ISDIR(a.permissions); }
            bool IsSymLink() const          { return LIBSSH2_SFTP_S_ISLNK(a.permissions); }
            bool IsSpecial() const          { return LIBSSH2_SFTP_S_ISCHR(a.permissions); }
            bool IsBlock() const            { return LIBSSH2_SFTP_S_ISBLK(a.permissions); }
            bool IsFifo() const             { return LIBSSH2_SFTP_S_ISFIFO(a.permissions); }
            bool IsSocket() const           { return LIBSSH2_SFTP_S_ISSOCK(a.permissions); }
            bool IsReadable() const         { return CanMode(IRUSR, IRGRP, IROTH); }
            bool IsWriteable() const        { return CanMode(IWUSR, IWGRP, IWOTH); }
            bool IsReadOnly() const         { return IsReadable() && !IsWriteable(); }
            bool IsExecutable() const       { return !IsDirectory() && CanMode(IXUSR, IXGRP, IXOTH); }

            DirEntry()                      { Zero(a); }
            DirEntry(const Nuller&)         { Zero(a); }

        private:
            bool CanMode(dword u, dword g, dword o) const;
            String file;
            String entry;
            SFtpAttrs a;
     };
    typedef Vector<DirEntry> DirList;

public:
    SFtp& Handle(SFtpHandle* hndl)  { if(hndl) handle = hndl; return *this; }
    SFtpSession* GetSession() const { return session; }
    SFtpHandle*  GetHandle() const  { return handle; }

    SFtp& StartOpen(const String& path, unsigned long flags, long mode);
    SFtp& StartClose();
    SFtp& StartRemove(const String& path);
    SFtp& StartRename(const String& oldpath, const String& newpath);
    SFtp& StartSetStat(const SFtpAttrs& attrs)                                       { StartFStat(const_cast<SFtpAttrs&>(attrs), true); return *this; }
    SFtp& StartGetStat(SFtpAttrs& attrs)                                             { Zero(attrs); StartFStat(attrs, false); return *this; }
    SFtp& StartSync();
    SFtp& StartOpenDir(const String& path);
    SFtp& StartListDir(DirList& list, Gate<DirEntry&> progress = Null);
    SFtp& StartListDir(const String& path, DirList& list, Gate<DirEntry&> progress = Null);
    SFtp& StartMakeDir(const String& path, long mode);
    SFtp& StartRemoveDir(const String& path);
    SFtp& StartMakeLink(const String& orig, const String& link)                      { StartSymLink(orig, const_cast<String*>(&link), LIBSSH2_SFTP_SYMLINK); return *this; }
    SFtp& StartReadLink(const String& path, String& target)                          { StartSymLink(path, &target, LIBSSH2_SFTP_READLINK); return *this;}
    SFtp& StartRealPath(const String& path, String& target)                          { StartSymLink(path, &target, LIBSSH2_SFTP_REALPATH); return *this; }

    SFtp& StartPut(const Stream& in, Gate<int64, int64> progress = Null);
    SFtp& StartPut(const Stream& in, const String& path, unsigned long flags,
                                long mode, Gate<int64, int64> progress = Null);
    SFtp& StartGet(Stream& out, Gate<int64, int64> progress = Null);
    SFtp& StartGet(Stream& out, const String& path, unsigned long flags, long mode,
                                Gate<int64, int64> progress = Null);

    bool Open(const String& path, unsigned long flags, long mode)                   { Clear(); StartOpen(path, flags, mode); return Execute(); }
    bool Close()                                                                    { Clear(); StartClose(); return Execute(); }
    bool Remove(const String& path)                                                 { Clear(); StartRemove(path); return Execute(); }
    bool Rename(const String& oldpath, const String& newpath)                       { Clear(); StartRename(oldpath, newpath);return Execute(); }
    bool SetStat(const SFtpAttrs& attrs)                                            { Clear(); StartSetStat(attrs); return Execute(); }
    bool GetStat(SFtpAttrs& attrs)                                                  { Clear(); StartGetStat(attrs); return Execute(); }
    bool Sync()                                                                     { Clear(); StartSync(); return Execute(); }
    bool OpenDir(const String& path)                                                { Clear(); StartOpenDir(path); return Execute(); }
    inline String GetDir() const                                                    { return dir; }
    bool ListDir(SFtp::DirList& list, Gate<DirEntry&> progress = Null)              { Clear(); StartListDir(list, progress); return Execute(); }
    bool ListDir(const String& path, DirList& list, Gate<DirEntry&> progress = Null){ Clear(); StartListDir(path, list, progress); return Execute(); }
    bool MakeDir(const String& path, long mode)                                     { Clear(); StartMakeDir(path, mode); return Execute(); }
    bool RemoveDir(const String& path)                                              { Clear(); StartRemoveDir(path); return Execute(); }
    bool MakeLink(const String& orig, const String& link)                           { Clear(); StartMakeLink(orig, link); return Execute(); }
    bool ReadLink(const String& path, String& target)                               { Clear(); StartReadLink(path, target); return Execute(); }
    bool RealPath(const String& path, String& target)                               { Clear(); StartRealPath(path, target); return Execute(); }

    bool Put(const Stream& in, Gate<int64, int64> progress = Null)                  { Clear(); StartPut(in, progress); return Execute(); }
    bool Put(const Stream& in, const String& path, unsigned long flags,
                            long mode, Gate<int64, int64> progress = Null)          { Clear(); StartPut(in, path, flags, mode, progress); return Execute(); }
    bool Get(Stream& out, Gate<int64, int64> progress = Null)                       { Clear(); StartGet(out, progress); return Execute(); }
    bool Get(Stream& out, const String& path, unsigned long flags,
                            long mode, Gate<int64, int64> progress = Null)          { Clear(); StartGet(out, path, flags, mode, progress); return Execute(); }

    bool Seek(int64 position);
    int64 Tell();

    SFtp();
    SFtp(Ssh& session) : SFtp() { Session(session); }
    virtual ~SFtp() { CleanUp(); }

protected:
    virtual void StartInit() override;
    virtual void StartStop() override;
    virtual void CleanUp()   override;

private:
    void StartFStat(SFtpAttrs& attrs, bool set);
    void StartSymLink(const String& path, String* target, int type);

    SFtpSession* session;
    SFtpHandle*  handle;
    DirEntry     dir_entry;
    String       dir;
};

class Channel : public Ssh::Subsystem {
public:
    SshChannel* GetChannel() const { return channel; }

    Channel&    StartRequest(const String& request, const String& params = Null);
    Channel&    StartExec(const String& cmdline);
    Channel&    StartShell();
    Channel&    StartSubsystem(const String& subsystem);
    Channel&    StartTerminal(const String& term, int width, int height);
    Channel&    StartSetEnv(const String& variable, const String& value);
    Channel&    StartRead(Stream& in, int64 size, Gate<int64, int64> progress = Null);
    Channel&    StartWrite(Stream& out, int64 size, Gate<int64, int64> progress = Null);
    Channel&    StartClose();
    Channel&    StartCloseWait();
    Channel&    StartSendEof();
    Channel&    StartRecvEof();
    Channel&    StartSendRecvEof();
    Channel&    StartReadStdErr(Stream& err);
    Channel&    StartWriteStdErr(Stream& err);
    Channel&    StartGetExitCode();
    Channel&    StartGetExitSignal();

    bool        Request(const String& request, const String& params = Null)        { Clear(); StartRequest(request, params); return Execute(); }
    bool        Exec(const String& cmdline)                                        { Clear(); StartExec(cmdline); return Execute(); }
    bool        Shell()                                                            { Clear(); StartShell(); return Execute(); }
    bool        Subsystem(const String& subsystem)                                 { Clear(); StartSubsystem(subsystem); return Execute(); }
    bool        Terminal(const String& term, int width, int height)                { Clear(); StartTerminal(term, width, height); return Execute(); }
    bool        SetEnv(const String& variable, const String& value)                { Clear(); StartSetEnv(variable, value); return Execute(); }
    bool        Read(Stream& in, int64 size, Gate<int64, int64> progress = Null)   { Clear(); StartRead(in, size, progress); return Execute(); }
    bool        Write(Stream& out, int64 size, Gate<int64, int64> progress = Null) { Clear(); StartWrite(out, size, progress); return Execute(); }
    bool        Close()                                                            { Clear(); StartClose(); return Execute(); }
    bool        CloseWait()                                                        { Clear(); StartCloseWait(); return Execute(); }
    bool        SendEof()                                                          { Clear(); StartSendEof(); return Execute(); }
    bool        RecvEof()                                                          { Clear(); StartRecvEof(); return Execute(); }
    bool        SendRecvEof()                                                      { Clear(); StartSendRecvEof(); return Execute(); }
    bool        ReadStdErr(Stream& err)                                            { Clear(); StartReadStdErr(err); return Execute(); }
    bool        WriteStdErr(Stream& err)                                           { Clear(); StartWriteStdErr(err); return Execute(); }
    int         GetExitCode();
    String      GetExitSignal();

    Channel();
    Channel(Ssh& session) : Channel() { Session(session); }
    virtual ~Channel() { CleanUp(); }

protected:
    virtual void StartInit() override;
    virtual void StartStop() override;
    virtual void CleanUp()   override;

    bool DataRead(int id, Stream& out, int64 size, Gate<int64, int64> progress = Null);
    bool DataWrite(int id, Stream& in, int64 size, Gate<int64, int64> progress = Null);

    SshChannel*     channel;
    int             code;
    StringBuffer    signal;
};

class Scp : public Channel {
public:
    Scp&        StartGet(Stream& out, const String& path, Gate<int64, int64> progress = Null);
    Scp&        StartPut(Stream& in, const String& path, long mode, Gate<int64, int64> progress = Null);
    bool        Get(Stream& out, const String& path, Gate<int64, int64> progress = Null)                 { Clear(); StartGet(out, path, progress); return Execute(); }
    bool        Put(Stream& in, const String& path, long mode, Gate<int64, int64> progress = Null)       { Clear(); StartPut(in, path, mode, progress); return Execute(); }
    inline bool operator()(Stream& out, const String& path, Gate<int64, int64> progress = Null)          { return Get(out, path, progress); }
    inline bool operator()(Stream& in, const String& path, long mode, Gate<int64, int64> progress = Null){ return Put(in, path, mode, progress); }

    Scp()                     { ClearStat(); type = Type::SCP; }
    Scp(Ssh& session) : Scp() { Session(session); }
    virtual ~Scp()            { ClearStat(); }

private:
    inline void ClearStat()   { Zero(file_stat); }
    libssh2_struct_stat file_stat;
};

class Exec : public Channel {
public:
    Exec&       StartExecute(const String& cmd, Stream& out, Stream& err);
    int         Execute(const String& cmd, Stream& out, Stream& err);
    inline int  operator()(const String& cmd, Stream& out, Stream& err)    { return Execute(cmd, out, err); }
    Exec()                      { type = Type::EXEC; }
    Exec(Ssh& session) : Exec() { Session(session); }
    virtual ~Exec() {}

private:
    void GetResult();
};

class KnownHosts : NoCopy {
public:
    struct Info {
        String key;
        int    type;
        int    status;

        bool   IsRSA() const        { return type == LIBSSH2_KNOWNHOST_KEY_SSHRSA;  }
        bool   IsDSS() const        { return type == LIBSSH2_KNOWNHOST_KEY_SSHDSS;  }
        bool   IsUnknown() const    { return !IsRSA() && !IsDSS(); }

        bool   IsFailure() const    { return status == LIBSSH2_KNOWNHOST_CHECK_FAILURE;  }
        bool   IsNotFound() const   { return status == LIBSSH2_KNOWNHOST_CHECK_NOTFOUND; }
        bool   IsMismatch() const   { return status == LIBSSH2_KNOWNHOST_CHECK_MISMATCH; }
        bool   IsMatch() const      { return status == LIBSSH2_KNOWNHOST_CHECK_MATCH;    }
    };

public:
    bool       Add(const String& host, int port, const Info& info, const String& comment);
    bool       Add(const String& host, const Info& info, const String& comment);
    bool       Remove(SshHost* host);
    bool       Load(const String& filename);
    bool       Save();
    bool       SaveAs(const String& filename);
    Info       Check(const String& host, int port);

    Vector<SshHost*> GetHosts();

    int        GetError() const               { return hosts_error.Get<int>();    }
    String     GetErrorDesc() const           { return hosts_error.Get<String>(); }

    KnownHosts(Ssh& session);
    virtual ~KnownHosts();

private:
    bool       Error();
    Ssh*       ssh;
    String     file_path;
    Tuple<int,String>  hosts_error;
    LIBSSH2_KNOWNHOSTS* hosts;
};

// U++ style memory (de/re)allocators for libssh2.
void* ssh_malloc(size_t size, void **abstract);
void* ssh_realloc(void *ptr, size_t size, void **abstract);
void  ssh_free(void *ptr, void **abstract);

// libssh2 error handler.
Tuple<int, String> ssh_liberror(SshSession* session, int code, const char* msg);

// libssh2 keyboard (interactive) authentication callback.
void ssh_keyboard_callback(const char *name, int name_len, const char *instruction, int instruction_len,
    int num_prompts, const LIBSSH2_USERAUTH_KBDINT_PROMPT *prompts, LIBSSH2_USERAUTH_KBDINT_RESPONSE *responses,
    void **abstract);
}
#endif
